/*
  Uygulama Adı  : İnternetten WEB tarayıcı üzerinden ayarlı termostat uygulaması
  Açıklama      : NodeMCU V3 ESP8266 ESP-12E STA (station: istasyon)
                  çalışma modundayken internetten web tarayıcı üzerinden
                  termostat (sıcaklık ayarlama) uygulaması
                  Ayarlanan sıcaklık değeri ESP8266'nın flash hafızasında
                  saklanır. Elektrik kesintilerinden sonra flash hafızadaki 
                  dosya okunarak alınan sıcaklık bilgisine göre ısıtıcı kontrol edilir.
  Tarih         : 26 Ağustos 2023
  Yazan         : İlhan DEMİR
*/

#include <ESP8266WiFi.h>
#include <ESP8266WebServer.h>
#include <FS.h>   //SPI ile haberleşen flash hafıza çipleri için SPIFFS kütüphanesi ekle.

//ESP8266'nın bağlanacağı kablosuz ağ bilgilerini tanımla.
const char *ssid = "kablosuz_ag_adiniz";
const char *sifre = "kablosuz_ag_sifreniz";

//sunucu isimli server tanımla. Web tarayıcılar için port numarası 80.
ESP8266WebServer sunucu(80);
//IP adresi, ağ geçidi ve alt ağ maskesi tanımla.
IPAddress yerelIP(192, 168, 1, 32);
IPAddress agGecidi(192, 168, 1, 1);
IPAddress altAgMaskesi(255, 255, 255, 0);
/*Windows cmd komut satırında ipconfig programını kullanarak
  ağ geçidi, alt ağ maskesi bilgilerini öğrenebilirsiniz.
  İnternetten ESP8266'ya ulaşmak için modem arayüzüne girilerek port
  açma/yönlendirme işlemi yapılmalıdır. Bu işlemden sonra farklı ağlar
  üzerinden web tarayıcıdan ISP tarafından modeme atanan genel IP adresiyle
  istek yapılırsa ESP8266'ya yönlendirme yapılır. Kodda tanımlanan
  "Ayarlı Termostat Uyg" başlıklı web sayfasına ulaşılarak istenen kontrol yapılabilir.
*/
#define rolePin D7
#define sensorPin A0
unsigned int sensorToplamDeger = 0;
unsigned int sensorOrtalamaDeger = 0;
float gerilim = 0;
float sicaklik = 0;
String ayarlananDeger = "24";
byte isitici = 0;

void setup()
{
  pinMode(rolePin, OUTPUT);
  Serial.begin(115200);       //Seri haberleşme hızını 115200 baud ayarla.

  SPIFFS.begin();                                  //Dosya sistemini başlat.
  if (!SPIFFS.exists("kayit.txt"))SPIFFS.format(); //Flash hafızada kayit.txt isimli bir dosya yoksa dosya sistemini formatla.
  else
  {
    File dosya = SPIFFS.open("kayit.txt", "r");    //kayit.txt isminde bir dosya mevcutsa dosyayı aç.
    ayarlananDeger = dosya.readStringUntil('\n');  //Dosyadan ilk satırı oku ve ayarlananDeger değişkenine yükle.
  }

  Serial.println();
  Serial.print("İstasyon yapılandırılıyor...");

  WiFi.mode(WIFI_STA);       //ESP8266'yı STA (station: istasyon) çalışma moduna al.
  WiFi.config(yerelIP, agGecidi, altAgMaskesi); //IP, ağ geçidi ve alt ağ maskesi adreslerini belirle.
  WiFi.begin(ssid, sifre);   //Wifi SSID ve şifre bilgilerini tanımla.

  Serial.println();
  Serial.println ("Bağlantı kuruluyor.");
  while (WiFi.status() != WL_CONNECTED) //Wi-Fi bağlantısı kurulana kadar bekle.
  {
    //Bağlantı sağlanana kadar her 500 msn'de bir . (nokta) karakteri gönder.
    delay(500);
    Serial.print(".");
  }

  Serial.println();
  Serial.println ("Bağlantı kuruldu.");
  Serial.print ("Kablosuz ağ adı = ");
  Serial.println (ssid);
  Serial.print("IP Adresi = ");
  Serial.println(WiFi.localIP());

  sunucu.on("/", anasayfa);           //Web tarayıcıya IP adresi yazılırsa anasayfa() fonksiyonunu çalıştır.
  sunucu.on("/veri", sicaklikGonder); // /veri linki geldiğinde sicaklikGonder() fonksiyonunu çalıştır.
  sunucu.onNotFound(sayfa_bulunamadi);//Bilinmeyen bir istek gelirse sayfa_bulunamadi() fonksiyonunu çalıştır.

  sunucu.begin();                     //Sunucuyu başlat.
  Serial.println();
  Serial.println("HTTP sunucu başlatıldı.");

  sicaklikOlc();                      //Sıcaklığı ölç ve ayarlanan sıcaklık değerine göre ısıtıcıyı kontrol et.
}

void loop()
{
  sunucu.handleClient();           //İşlemlerin sürekli kontrolü için handleClient fonksiyonunu çalıştır.
}

void sayfa_bulunamadi()            //Hatalı bir istek yapılırsa bu fonksiyon çalışır.
{
  String sayfa = "<!doctype html>";//Html sayfası oluştur. String tipinde sayfa değişkenine yükle.
  sayfa += "<html>";               //HTML bilmiyorsanız çevrimiçi HTML düzenleyiciden yararlanabilirsiniz.
  sayfa += "<head>";
  sayfa += "<meta charset=UTF-8>";
  sayfa += "<title>Ayarlı Termostat Uyg</title>";
  sayfa += "</head>";
  sayfa += "<body>";
  sayfa += "<h2>404 - Aradığınız sayfa bulunamadı.</h2>";
  sayfa += "<p><a href= \"/\"><strong>Ana Sayfa</strong></a></p>";
  sayfa += "</body>";
  sayfa += "</html>";

  sunucu.send(200, "text/html; charset=utf-8", sayfa); //Web sayfasını istek yapılan tarayıcıya gönder.
  //sunucu.send(404, "text/html; charset=utf-8", "<html><h1>404 - Aradığınız sayfa bulunamadı.</h1></html>"); //Ekrana hata mesajı yazdır.
}

void anasayfa()                    //IP adresi ile istek yapılırsa bu fonksiyon çalışır.
{
  sicaklikOlc();

  String sayfa = "<!doctype html>";//Html sayfası oluştur. String tipinde sayfa değişkenine yükle.
  sayfa += "<html>";
  sayfa += "<head>";
  sayfa += "<meta charset=UTF-8>";
  sayfa += "<title>Ayarlı Termostat Uyg</title>";
  sayfa += "<meta http-equiv=\"refresh\" content=\"10; url=/\">"; //10 sn'de bir ana sayfayı yeniden yükle.
  sayfa += "</head>";
  sayfa += "<body>";
  sayfa += "<h2><span style=\"font-family: terminal, monaco, monospace;\"><span style=\"color: #3598db;\"><em><strong>Ayarlı Termostat Uygulaması</strong></em></span></h2>";
  sayfa += "<p>&nbsp;</p>";

  sayfa += "<form action=\"/veri\">";
  //min değeri 0, max değeri 40 ve artış/azalış adım değeri 0,1 olan ayarlanansicaklik isminde argüman oluştur.
  sayfa += "<p><span style=\"font-family: verdana, geneva, sans-serif;\">Sıcaklığı  ";
  sayfa += "<input type=\"number\" name=\"ayarlanansicaklik\" min=\"0\" max=\"40\"  step=\"0.1\" value=\"";
  sayfa += ayarlananDeger.toFloat();
  sayfa += "\" style=\"width: 4em\">";
  //ayarlanansicaklik değerini Ayarla butonuyla sicaklikGonder sayfasına aktar.
  sayfa += "<input type=\"submit\" value=\"Ayarla\"></form>";
  sayfa += "<p><span style=\"font-family: verdana, geneva, sans-serif;\">Sıcaklık = ";
  sayfa += sicaklik; //Sıcaklık değerini web sayfasında göster.
  sayfa += "°C";
  sayfa += "<p><span style=\"font-family: verdana, geneva, sans-serif;\"><span style=\"color: #e03e2d;\">Isıtıcı";
  sayfa += isitici ? " çalışıyor." : " çalışmıyor.";  //Isıtıcının durumunu göster.

  sayfa += "<h5><span style = \"font-family: terminal, monaco, monospace;\"><em><span style=\"color: #3598db;\">&copy; www.beyzemiter.com & İlhan DEMİR</span></em></span></h5>";
  sayfa += "</body>";
  sayfa += "</html>";

  sunucu.send(200, "text/html; charset=utf-8", sayfa); //Web sayfasını istek yapılan tarayıcıya gönder.
}

void sicaklikGonder()                        //IP adresi /veri linkiyle istek yapılırsa bu fonksiyon çalışır.
{
  ayarlananDeger = sunucu.arg("ayarlanansicaklik");   //ayarlanansicaklik isimli argümanın değerini String tipinde ayarlananDeger değişkenine yükle.
  File dosya = SPIFFS.open("kayit.txt", "w");         //Flash hafızada bulunan kayit.txt dosyasını yazma modunda aç.
  dosya.println(ayarlananDeger);                      //Dosyanın ilk satırına webden alınan ayarlananDeger değişkenini yazdır.
  sicaklikOlc();  //Sıcaklık ölçümü yap ve ayarlanan sıcaklık değerine göre ısıtıcıyı kontrol et.
  Serial.println("Ayarlanan Sıcaklık Değeri = " + ayarlananDeger + "°C"); //ayarlananDeger değişkenini seri port ekranında yazdır.

  String sayfa = "<!doctype html>";//Html sayfası oluştur. String tipinde sayfa değişkenine yükle.
  sayfa += "<html>";
  sayfa += "<head>";
  sayfa += "<meta charset=UTF-8>";
  sayfa += "<title>Ayarlı Termostat Uyg</title>";
  sayfa += "<meta http-equiv=\"refresh\" content=\"10; url=/\">"; //10 sn'de bir ana sayfayı yeniden yükle.
  sayfa += "</head>";
  sayfa += "<h2><span style=\"font-family: terminal, monaco, monospace;\"><span style=\"color: #3598db;\"><em><strong>Ayarlı Termostat Uygulaması</strong></em></span></h2>";
  sayfa += "<p>&nbsp;</p>";

  sayfa += "<form action=\"/veri\">";
  //min değeri 0, max değeri 40 ve artış/azalış adım değeri 0,1 olan ayarlanansicaklik isminde argüman oluştur.
  sayfa += "<p><span style=\"font-family: verdana, geneva, sans-serif;\">Sıcaklığı  ";
  sayfa += "<input type=\"number\" name=\"ayarlanansicaklik\" min=\"0\" max=\"40\"  step=\"0.1\" value=\"";
  sayfa += ayarlananDeger.toFloat();
  sayfa += "\" style=\"width: 4em\">";
  //ayarlanansicaklik değerini Ayarla butonuyla sicaklikGonder sayfasına aktar.
  sayfa += "<input type=\"submit\" value=\"Ayarla\"></form>";
  sayfa += "<p><span style=\"font-family: verdana, geneva, sans-serif;\">Sıcaklık = ";
  sayfa += sicaklik; //Sıcaklık değerini web sayfasında göster.
  sayfa += "°C";
  sayfa += "<p><span style=\"font-family: verdana, geneva, sans-serif;\"><span style=\"color: #e03e2d;\">Isıtıcı";
  sayfa += isitici ? " çalışıyor." : " çalışmıyor.";  //Isıtıcının durumunu göster.

  sayfa += "<p> <a href=\"/\"><strong>Ana Sayfa</strong></a></p>";
  sayfa += "</body>";
  sayfa += "</html>";

  sunucu.send(200, "text/html; charset=utf-8", sayfa);        //Web sayfasını istek yapılan tarayıcıya gönder.
}

void sicaklikOlc()
{
  unsigned int sensorToplamDeger = 0, sensorOrtalamaDeger = 0;
  for (int i = 0; i < 50; i++)
  {
    sensorToplamDeger += analogRead (sensorPin);  //Sensörü 50 kez oku ve okunan değerleri topla.
  }
  sensorOrtalamaDeger = sensorToplamDeger / 50;   //Sensörden okunan değerlerin ortalamasını al.
  gerilim = (sensorOrtalamaDeger / 1023.0) * 3300;//Sensörden alınan bilgiyi gerilime dönüştür. Referans voltaj aralığı 0-3,3V.
  sicaklik = gerilim / 10.0;                      //Gerilimi dereceye dönüştür.

  if (sicaklik >= (ayarlananDeger.toFloat()))     //Sıcaklık ayarlanan değerden büyük eşit mi?
  {
    digitalWrite(rolePin, LOW);                   //Isıtıcıyı kapat.
    isitici = 0;
  }
  else
  {
    digitalWrite(rolePin, HIGH);                  //Isıtıcıyı aç.
    isitici = 1;
  }
}
